// Copyright ® 2025 OneSpan North America, Inc. All rights reserved. 

 
/////////////////////////////////////////////////////////////////////////////
//
//
// This file is example source code. It is provided for your information and
// assistance. See your licence agreement for details and the terms and
// conditions of the licence which governs the use of the source code. By using
// such source code you will be accepting these terms and conditions. If you do
// not wish to accept these terms and conditions, DO NOT OPEN THE FILE OR USE
// THE SOURCE CODE.
//
// Note that there is NO WARRANTY.
//
//////////////////////////////////////////////////////////////////////////////


package com.vasco.orchestration.sample;

import android.os.Build;
import android.os.Bundle;
import android.view.View;
import android.view.ViewGroup;
import android.view.Window;
import androidx.annotation.IdRes;
import androidx.annotation.LayoutRes;
import androidx.annotation.Nullable;
import androidx.appcompat.app.AppCompatActivity;
import androidx.core.view.ViewCompat;
import androidx.core.view.WindowInsetsCompat.Type;
import androidx.core.graphics.Insets;

/**
 * Abstract base activity to enable edge-to-edge layout and handle system window insets (status bar, navigation bar).
 * <p>
 * Subclasses should override {@link #getRootLayoutId()} to apply insets automatically to the root view.
 */
public abstract class BaseEdgeToEdgeActivity extends AppCompatActivity {

    /**
     * Returns the ID of the root layout that should receive system window insets.
     *
     * @return the view ID of the root layout
     */
    @IdRes
    protected abstract int getRootLayoutId();

    /**
     * Called when the activity is starting. Enables edge-to-edge behavior and applies status bar icon settings.
     *
     * @param savedInstanceState If the activity is being re-initialized after previously being shut down, this Bundle contains the most recent data.
     */
    @Override
    protected void onCreate(@Nullable Bundle savedInstanceState) {
        applyEdgeToEdge(getWindow());
        super.onCreate(savedInstanceState);
    }

    /**
     * Sets the activity content from a layout resource. Also applies window insets to the root layout.
     *
     * @param layoutResID Resource ID to be inflated
     */
    @Override
    public void setContentView(@LayoutRes int layoutResID) {
        super.setContentView(layoutResID);
        applyInsetsToRoot();
    }

    // Internal method to enable edge-to-edge drawing
    private void applyEdgeToEdge(Window window) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
            window.setDecorFitsSystemWindows(false);
        } else {
            View decorView = window.getDecorView();
            decorView.setSystemUiVisibility(
                    View.SYSTEM_UI_FLAG_LAYOUT_STABLE | View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN);
        }
    }

    // Internal method to apply padding for system insets to the root layout
    private void applyInsetsToRoot() {
        final View rootView = findViewById(getRootLayoutId());
        if (rootView == null) return;

        ViewCompat.setOnApplyWindowInsetsListener(rootView, (view, insets) -> {
            final Insets systemBars = insets.getInsets(Type.systemBars());

            ViewGroup.MarginLayoutParams layoutParams = (ViewGroup.MarginLayoutParams) rootView.getLayoutParams();
            layoutParams.topMargin = systemBars.top;
            layoutParams.bottomMargin = systemBars.bottom;
            layoutParams.leftMargin = systemBars.left;
            layoutParams.rightMargin = systemBars.right;

            rootView.setLayoutParams(layoutParams);

            return insets;
        });
    }
}
